/*
 * cs101 SocketBroker Utility
 * Figures out where the server is and connects.
 * $Id: SocketBroker.java,v 1.2 1998/06/03 21:01:25 tparnell Exp $
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see <a href="http://www.ai.mit.edu/projects/cs101">the
 * CS101 homepage</a> or email <las@ai.mit.edu>.
 *
 * Copyright (C) 1996 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */

package cs101.util;
import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;

/**
 * Figures out where the server is (with the users help) and connects.
 * <p>
 * All dialogs are done with Frames b/c applets are not Frames, and
 * therefore can't "parent" Dialog boxes.
 * 
 * @author Todd C. Parnell, tparnell@ai.mit.edu
 * @author Joshua R. Brown, reuben@ai.mit.edu
 * @version $Id: SocketBroker.java,v 1.2 1998/06/03 21:01:25 tparnell Exp $
 * <br>
 * Copyright 1996 Massachusetts Institute of Technology
 *
 */
public class SocketBroker {
  
    
  /**
   * Calls getConnection(applet, "SERVER", "PORT")
   * 
   * @param applet the applet that wants a connection
   * @see #getConnectoin(java.applet.Applet, String, String);
   * @return a socket connected to the discovered server
   */
  public Socket getConnection(Applet applet) {   
    return getConnection(applet, "SERVER", "PORT");    
  }


  /**
   * Tries to get the server and port from the applet parameters passed in.  
   * If server parameter is not defined, gives user the option of using
   * the host that the applet was downloaded from.
   * Calls the approptiate getConnection method for the information it finds.
   *
   * @param applet the applet that wants a connection
   * @param hostParamName the name of the host parameter to look for
   * @param portParamName the name of the port parameter to look for
   * @see   #getConnection()
   * @see   #getConnection(java.lang.String, int)
   * @see   #getConnection(int)
   * @see   #getConnection(java.lang.String)
   * @return a Socket connected to that server.
   */
  public Socket getConnection(Applet applet, String hostParamName,
				     String portParamName) {

    System.out.println("Checking for applet parameters " +
		       hostParamName+" and "+portParamName+" ...");

    // try to get host from the applet
    String host = applet.getParameter(hostParamName);    
    
    // if host is not defined
    if (host == null) {
      
      // bring up a dialog 
      AppletHostDialog ahd = new AppletHostDialog(this);
      
      // wait for an answer
      synchronized (this) {
	try { wait(); }
	catch (InterruptedException e) {}
      }
      
      // if answer is yes (true) use doc base
      if (ahd.answer) {
	System.out.println("Using applet's web server as server host ...");
	host = applet.getDocumentBase().getHost();
      }
      
      ahd = null;

    }
    
    // try to get port from the applet
    String port = applet.getParameter(portParamName);
    
    // call the appropriate function and return what it return
    if (port != null) {
      int portNum = Integer.parseInt(port);  
      if (host != null) 
	return getConnection(host, portNum);
      else 
	return getConnection(portNum);
    }
    else if (host != null) 
      return getConnection(host);
    else
      return getConnection();	
  }


  /**
   * Brings up dialogs to get the server's host and port from the user.
   * Calls connect when it gets good info.
   *
   * @return a Socket connected to that server.
   * @see #connect
   */
  public Socket getConnection() {    
    
    InetAddress address = getAddress();    
    int port = getPort();

    return connect(address, port);

  }


  /**
   * Checks the validity of the host passed in. 
   * Brings up a dialog to get port.
   * Calls connect when it gets good info.
   * 
   * @param host the name of the host to connect to.
   * @return a Socket connected to that server.
   * @see #connect
   */
   public Socket getConnection(String host) {
    
    // get address of host
    InetAddress address = null;
    try { address = InetAddress.getByName(host); }
    catch (UnknownHostException e) {
      System.out.println("SocketBroker: Invalid hostname " + host);
      System.out.println(e);
      return getConnection();
    }

    int port = getPort();

    return connect(address, port);
   
   }


  /**
   * Brings up a dialog to get port.
   * Calls connect when it gets good info.
   * 
   * @param address the address of the host to connect to.
   * @see #connect
   * @return a Socket connected to that server.
   */
  public Socket getConnection(InetAddress address) {
    
    int port = getPort();
    
    return connect(address, port);

  }
 

  /**
   * Checks the validity of the port passed in. 
   * Brings up a dialog to get host.
   * Calls connect when it gets good info.
   * 
   * @see #connect
   * @param port the number of the port to connect to.
   * @return a Socket connected to that server.
   */
  public Socket getConnection(int port) {

    InetAddress address = getAddress();
    
    // check port number
    if (port <= 1024 && port > 9999) {
      System.out.println("SocketBroker: Invalid port number "+port+
			 " received.");
      System.out.println("Port numbers must be between 1024 and 9999.");
      port = getPort();
    }

    return connect(address, port);
    
  }


  /**
   * Checks the validity of the host and port passed in. 
   * Calls connect when it gets good info.
   *
   * @see #connect
   * @param host the name of the host to connect to.
   * @param port the number of the port to connect to.
   * @return a Socket connected to that server.
   */
  public Socket getConnection(String host, int port) {
    
    // get address of host
    InetAddress address;
    try { address = InetAddress.getByName(host); }
    catch (UnknownHostException e) {
      System.out.println("SocketBroker: Invalid hostname " + host);
      System.out.println(e);
      return getConnection(port);
    }

    // check the port number given
    if (port <= 1024 && port > 9999) {
      System.out.println("SocketBroker: Invalid port number "+port+
			 " received.");
      System.out.println("Port numbers must be between 1024 and 9999.");
      port = getPort();
    }
     
    // if we got here all clear to connect
    return connect(address, port);
    
  }


  /**
   * Brings up a dialog to get the host name from the user.
   * Tries to convert the hostname to an address.
   * Repeats on failure.
   *
   * @return the address of the host recieved
   */
  private InetAddress getAddress() {
    boolean fail = false;
    InetAddress address = null;
    
    do {
      // bring up a dialog to get the host name
      HostDialog hd = new HostDialog(this);
      // wait for an answer
      synchronized (this) {
	try { wait(); }
	catch (InterruptedException e) {}
      }      
      String host = hd.answer;
      try { address = InetAddress.getByName(host); }
      catch (UnknownHostException e) {
	System.out.println("SocketBroker: Invalid hostname " + host);
	System.out.println(e);
	fail = true;
      }
      hd = null;
    } while (fail);    

    return address;

  }


  /**
   * Brings up a dialog to get the host name from the user.
   * Checks the validity of the port number recieved.
   * Repeats on invalid input.
   *
   * @return the port number received.
   */
  private int getPort() {

    while(true) {
      // bring up a dialog to get the port number - until it's right
      PortDialog pd = new PortDialog(this);
      // wait for an answer
      synchronized (this) {
	try { wait(); }
	catch (InterruptedException e) {}
      }      	
      int port = pd.port;
      pd.dispose();
      pd = null;
      
      if (port > 1024 && port <= 9999)
	return port;

      System.out.println("SocketBroker: Invalid port number "+port+
			 " received.");
      System.out.println("Port numbers must be between 1024 and 9999.");
	
    }

  }


  /**
   * Actually creates a socket connected to the addess and port
   * passed in.  Calls getConnection() on failure in order to 
   * start the whole process over again.
   *
   * @see #getConnection()
   * @return a Socket connected to that server.
   */
  private Socket connect(InetAddress address, int port) {

    // for apperance only
    String host = address.getHostName();
    if (host == null)
      host = "localhost";

    System.out.println("Trying port " + port + " on host "+host+" ...");
    try { return new Socket(address, port); }
    catch (IOException e) {
      if (e.toString().equals("java.net.SocketException: Broken pipe")) {
	System.out.println("SocketBroker: Server not found at port "+
			   port+" on host "+host+".");
	System.out.println("Make sure server is running and try again.");
      }
      else 
	System.out.println("SocketBroker: Error durring connect: " + e);
      // go back to the start
      return getConnection();       
    }     

  }
  
}  


class AppletHostDialog extends Frame {

  private final Button yes;
  private final Button no;
  private final Object toNotify;

  public boolean answer = true;

  public AppletHostDialog(Object toNotify) {
    super("Applet Host Dialog");
    
    this.setSize(400,200);
    this.setCursor(new Cursor(Cursor.HAND_CURSOR));

    this.toNotify = toNotify;
    
    Panel questions = new Panel();
    questions.add(new Label("Connect to the host that the applet "+
			    "was downloaded from?"));
    
    yes = new Button("yes");
    no = new Button("no");

    yes.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	setVisible(false);
	answer = true;
	notifyme();
      }
    });

    this.addFocusListener(new FocusListener() {
      public void focusGained(FocusEvent e) {
	yes.requestFocus();
      }

      public void focusLost(FocusEvent e) {}
    });

    no.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	setVisible(false);
	answer = false;
	notifyme();
      }
    });


    Panel buttons = new Panel();
    buttons.add(yes);
    buttons.add(no);

    this.add("Center",questions);    
    this.add("South", buttons);

    this.pack();
    this.show();

  }

  void notifyme() {
    synchronized (toNotify) { toNotify.notify(); }
  }

}



class HostDialog extends Frame {

  private final TextField hostName;
  private Object toNotify;

  public String answer = null;

  public HostDialog(Object toNotify) {
    super("Hostname Dialog");

    this.setSize(400,200);
    this.setCursor(new Cursor(Cursor.HAND_CURSOR));

    this.toNotify = toNotify;
    
    hostName = new TextField(40);
    hostName.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	String temp = e.paramString();
	setAnswer(temp);
	if (temp != null) {
	  setVisible(false);
	  dispose();
	  notifyme();
	}
      }
    });

    this.addFocusListener(new FocusListener() {
      public void focusGained(FocusEvent e) {
	hostName.requestFocus();
      }

      public void focusLost(FocusEvent e) {}
    });


    this.add("Center", new Label("Please enter the name of "+
                                 "the host to connect to:"));
    this.add("South", hostName);

    this.pack();
    this.show();
  }

  void notifyme() {
    synchronized (toNotify) { toNotify.notify(); }
  }

  void setAnswer(String text) { this.answer = text; }

}


class PortDialog extends Frame {

  private final TextField portNumber;
  private final Object toNotify;

  public String answer = null;
  public int port = 5000;

  public PortDialog(Object toNotify) {
    super("Port Number Dialog");

    this.setSize(500,200);
    this.setCursor(new Cursor(Cursor.HAND_CURSOR));

    this.toNotify = toNotify;
    
    portNumber = new TextField(8);
    portNumber.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	String temp = e.paramString();
	answer = temp;
	if (temp != null) {
	  port = Integer.parseInt(answer);
	  setVisible(false);
	  notifyme();
	}
      }
    });

    this.setLayout(new FlowLayout());
    this.add(new Label("Please enter the port number to connect to:"));
    this.add(portNumber);

    this.addFocusListener(new FocusListener() {
      public void focusGained(FocusEvent e) {
	portNumber.requestFocus();
      }

      public void focusLost(FocusEvent e) {}
    });

    this.pack();
    this.show();
  }

  void notifyme() {
    synchronized (toNotify) { toNotify.notify(); }
  }

}

/* Comments:
 *
 * History:
 *     $Log: SocketBroker.java,v $
 *     Revision 1.2  1998/06/03 21:01:25  tparnell
 *     update from Java 1.0 to 1.1
 *
 *     Revision 1.1  1998/03/13 22:18:22  tparnell
 *     Import from server crash.  I think the src and class files match up.
 *
 *     Revision 1.3  1996/08/07 21:39:16  reuben
 *     Fixed some javadoc comments.
 *
 *     Revision 1.2  1996/08/07 15:53:53  reuben
 *     Fixed bug in PortNumberDialog.
 *
 *     Revision 1.1  1996/08/07 15:26:25  reuben
 *     Moved SocketBroker to the cs101 package.
 *
 *
 */
