/*
 * Generic Server Application
 * $Id: Server.java,v 1.3 1998/06/24 16:18:24 tparnell Exp $
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see <a href="http://www.ai.mit.edu/projects/cs101">the
 * CS101 homepage</a> or email <las@ai.mit.edu>.
 *
 * Copyright (C) 1996 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */

package cs101.util;

import java.net.*;
import java.io.*;
import java.util.*;

/**
 * Generic Server class:  broadcasts whatever messages it receives to
 * all clients other than the message's originator.  Spawns one thread to
 * listen and multiple others to handle the clients as they connect.
 * <p>
 * Takes a single optional argument to specify the port. Otherwise
 * spawns a UI dialog to get connection information from the user.
 * <p>
 * Some hints taken from ORA Servers.
 *
 * @see      cs101.util.RunServer
 * @see      cs101.util.Client
 * @see      cs101.util.ServerDialog
 *
 * @author   Todd C. Parnell, tparnell@ai.mit.edu
 * @author   Maciej Stachowiak, maciej@ai.mit.edu
 * @author   Emil Sit, sit@mit.edu
 * @author   Joshua Reuben Brown, reuben@ai.mit.edu
 * @author   Lynn Andrea Stein, las@ai.mit.edu
 * @version  $Id: Server.java,v 1.3 1998/06/24 16:18:24 tparnell Exp $
 * <br>
 * Copyright 1996 Massachusetts Institute of Technology
 *
 */
public class Server implements Runnable {

  /** the server socket, i.e., the listener */
  protected ServerSocket sock;

  /** list of BabySitters to handle each client connection  */
  protected Vector babySitters;

  /** animating Thread */
  protected Thread spirit;

  /**
   * Creates a server socket.  Waits for connections.
   * No argument version prompts users for port information.
   */
  public Server () {
    ServerDialog sd = new ServerDialog();
    sd.ask();
    this.listenOn( sd.getPort() );
  }

  /**
   * Creates a server socket.  Waits for connections.
   *
   * @param port   The integer representing the port number to listen on.
   */
  public Server( int port ) {
    this.listenOn( port );
  }

  /**
   * Helper method to actually open the ServerSocket and intialize
   * other state.  Also spawns Thread to continue listening.  Called
   * internally. 
   *
   * @param port   The integer representing the port number to listen on.
   */
  protected void listenOn( int port ) {
    System.out.println("Server:  starting up on port " + port );
    try {
      this.sock = new ServerSocket( port );
    } catch (IOException e) {
      throw new RuntimeException("Server:  failed to listen on port "
+ port );
    }
    this.babySitters = new Vector();
    this.spirit = new Thread( this );
    this.spirit.start();
  }

  /**
   * Wait loop to service the ServerSocket.  Called by this server's
   * Thread.  Should not be called externally.
   * Each time that  a connection is recieved, forks a BabySitter to 
   * handle that connection.
   */
  public void run() {
    try {
      while ( true ) {
	System.out.println("Server:  listening for a connection");
	this.spawnBabySitter( this.sock.accept() );
      }
    } catch (IOException e) {
      System.out.println( "Server:  abrupt failure on accept attempt." );
      System.out.println( "Server:  no longer accepting connections." );
    }
  }
    
  /**
   * Removes a BabySitter for the list of babySitters.
   * 
   * @param bbst the BabySitter to be remooved from use.
   */
  protected void removeBabySitter(BabySitter bbst) {
    this.babySitters.removeElement(bbst);
    bbst.stop();
  }

  /** 
   * Creates a BabySitter with the client socket passed and
   * adds it to the list of BabySitters.
   *
   * @param s the client socket that this BabySitter will handle
   */
  protected void spawnBabySitter(Socket s) {
    BabySitter bbst=new BabySitter(s, this);
    babySitters.addElement(bbst);
  }

  /**
   * Sends the string passed in to all clients accept the one
   * represented by the BabySitter passed in.
   *
   * @param s the string to send
   * @param bbst the BabySitter that handle the client that we don't want to 
   *           send to (the client that the message came from).
   */
  protected void sendToAllExcept(String s, BabySitter bbst) {
    for(Enumeration e=babySitters.elements();e.hasMoreElements();) {
      BabySitter current = (BabySitter) e.nextElement();
      if (current != bbst) {
	current.send(s);
      }
    }
  }
}

/* Comments:
 *
 * History:
 *     $Log: Server.java,v $
 *     Revision 1.3  1998/06/24 16:18:24  tparnell
 *     more revisions from summer98
 *
 *     Revision 1.2  1998/06/03 21:32:08  tparnell
 *     conversion from Java 1.0 to 1.1
 *
 *     Revision 1.6  1996/11/18 17:10:24  las
 *     All files appear to be working.  Some of the files in this directory
 *     belong in cs101.util.  These will be moved Real Soon Now.  Also, added
 *     ClientMonitor, which watches net traffic but doesn't send.
 *
 *     Revision 1.5  1996/11/18 16:41:21  las
 *     Client, Server, their dependencies (e.g. the Dialogs and their
 *     invokers) all work.  ClientTester is a bit awkward in that it doesn't
 *     echo received strings until something is read.  Oh, well.
 *     Moving them to cs101.util in the next revision.
 *
 *     Revision 1.4  1996/11/17 22:28:16  las
 *     Everything compiles (now).  Client, Server, ClientDialog, ServerDialog,
 *     StringHandler, and RunServer need to be moved to cs101.util.  But
 *     first, to test....
 *
 *     Revision 1.3  1996/11/17 20:43:19  las
 *     Server, RunServer written, not tested.  Need to be moved to
 *     cs101.util....
 *
 *     Revision 1.2  1996/11/17 20:37:38  las
 *     Server is written, but not yet tested.  Should be used with RunServer.java.
 *
 *     Revision 1.1  1996/11/17 19:39:33  las
 *     Intermediate state on the way to an application version.  Some files
 *     here actually belong in CS101.  Otherwise, the general file structure
 *     is correct, but the file contents are in some cases grossly out of
 *     line.
 *
 *     Desired structure:
 *     SharedWhiteboardDemo runs the (client side of the) demo.
 *     SharedWhiteboard is a helper file that starts things up correctly;
 *     it's the class definition, though, not the main (which is in
 *     SharedWhiteboardDemo).
 *     NetScribbleData is ScribbleData that sets up a client connection.  It
 *     uses StringHandler and Client.
 *     StringHandler, ClientDialog, ServerDialog, and maybe Client and Server
 *     go in the cs101 package.  StringHandler is the interface Client uses.
 *     ClientDialog and ServerDialog are QueryDialogs specialized for
 *     obtaining the appropriate information.  Client and Server are pretty
 *     generic (Server is a broadcaster and Client has send and
 *     StringHandler.handleString.)
 *     Fall 96, they write SharedWhiteboard(Demo), NetScribbleData, and
 *     Client.
 *
 *     Revision 1.5  1996/08/01 18:26:31  reuben
 *     More javadoc tweaking (hopefully the final pass)
 *
 *     Revision 1.4  1996/08/01 01:56:34  reuben
 *     Yesturday's changes were tested.
 *     (I wonder if using cvs and $log$ creates a circular modifcation problem)
 *
 *     Revision 1.3  1996/07/30 17:26:00  reuben
 *     Added/corrected javadoc comments.
 *
 *     Revision 1.2  1996/07/23 21:10:38  reuben
 *     Moved Colorfield and DisplayField to cs101.util package.
 *     Made the necessary changes to PNC.
 *     Compiled and tested all 4 demos.
 *
 *     Revision 1.1.1.1  1996/07/18 17:38:24  sit
 *     Import from /mit/6.096/share/classes after 6.80s session
 *
 *     Revision 1.1  1996/07/02 21:47:51  las
 *     Initial revision
 *
 */

