/*
 * cs101 Line utility
 * $Id: Line.java,v 1.2 1998/06/04 18:53:58 tparnell Exp $
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see <a href="http://www.ai.mit.edu/projects/cs101">the
 * CS101 homepage</a> or email <las@ai.mit.edu>.
 *
 * Copyright (C) 1996 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */

package cs101.util;

import java.awt.Color;
import java.awt.Graphics;
import java.util.StringTokenizer;

/**
 * Implements a 4-coordinate (+ optional Color) line abstraction with
 * its own drawing method.  Also, implements a translation between
 * Line objects and String objects for handling by less intelligent
 * protocols.
 * 
 * @see      java.awt.Graphics
 *
 * @author   Maciej Stachowiak, maciej@ai.mit.edu
 * @author   Lynn Andrea Stein, las@ai.mit.edu
 * @version  $Id: Line.java,v 1.2 1998/06/04 18:53:58 tparnell Exp $
 * <br>
 * Copyright 1996 Massachusetts Institute of Technology
 *
 */
public class Line {
    private Color c;
    private int startX, startY, endX, endY;

    /**
     * This specifies the default color for a line.
     * It may not be changed.
     */
    public static final Color DEFAULT_COLOR = Color.black;

    /**
     * Construct a line, specifying all 4 coordinates and using
     * the default color.
     */
    public Line ( int startX, int startY, int endX, int endY ) {
      this( startX, startY, endX, endY, Line.DEFAULT_COLOR );
    }

    /**
     * Construct a line, specifying all 4 coordinates and a specific
     * color.
     *
     * @see java.awt.Color
     */
    public Line ( int startX, int startY, int endX, int endY, Color c ) {
      this.startX = startX;
      this.startY = startY;
      this.endX = endX;
      this.endY = endY;
      if (c == null) this.c = Line.DEFAULT_COLOR;
      else /* c != null */ this.c = c;
    }

    public void drawOn (Graphics g) {
      g.setColor(c);
      g.drawLine(this.startX, this.startY, this.endX, this.endY);
    }

  // The rest of this class is for use with primitive datahandlers
  // which can only cope with Strings.  It provides a utility package
  // which enables coercion between a particular String representation
  // and objects of the Line class.
    private Line() {}

    /**
     * Given a String that was originally created by packLine, this
     * function will return a reference to a new Line object that
     * represents the original Line.
     *
     * @see #packLine
     */
    public static Line extractLine( String s ) {
	StringTokenizer tokens = new StringTokenizer(s," \t\r\n:");
	Line l = new Line();
	l.startX = Integer.parseInt( tokens.nextToken() );
	l.startY = Integer.parseInt( tokens.nextToken() );
	l.endX = Integer.parseInt( tokens.nextToken() );
	l.endY = Integer.parseInt( tokens.nextToken() );
	if ( tokens.hasMoreTokens() ) {
	    l.c = new Color( Integer.parseInt( tokens.nextToken() ) );
	} else {
	    l.c = Line.DEFAULT_COLOR;
	}
	return l;
    }

    /**
     * This method converts a Line into a String.
     *
     * @see #extractLine
     */
    public static String packLine( Line l ) {
        return ( Integer.toString( l.startX )
		 + ":"
		 + Integer.toString( l.startY )
		 + ":"
		 + Integer.toString( l.endX )
		 + ":"
		 + Integer.toString( l.endY )
		 + ":"
		 + Integer.toString( l.c.getRGB() )
	         + "\n" );
    }
}


/* Comments:
 *
 * History:
 *     $Log: Line.java,v $
 *     Revision 1.2  1998/06/04 18:53:58  tparnell
 *     prevented passing in color = null from raising exception
 *
 *     Revision 1.1  1998/03/13 22:18:15  tparnell
 *     Import from server crash.  I think the src and class files match up.
 *
 *     Revision 1.4  1996/11/18 17:25:03  las
 *     Added revised SharedWhiteboard support classes.  These versions of
 *     Client and Server supercede the previous ones and are not directly
 *     backwards compatible.  In particular, Server is an instantiable class
 *     rather than a primarily static one (use RunServer to run it), and
 *     Client uses StringHandler rather than subclassing to specialize it.
 *     Line.java just picked up some obscure documentation along the way.
 *     Otherwise, classes are direct imports from SharedWhiteboard.
 *
 *     Revision 1.1  1996/11/18 17:10:15  las
 *     All files appear to be working.  Some of the files in this directory
 *     belong in cs101.util.  These will be moved Real Soon Now.  Also, added
 *     ClientMonitor, which watches net traffic but doesn't send.
 *
 *     Revision 1.3  1996/08/01 18:26:29  reuben
 *     More javadoc tweaking (hopefully the final pass)
 *
 *     Revision 1.2  1996/07/30 17:26:00  reuben
 *     Added/corrected javadoc comments.
 *
 *     Revision 1.1.1.1  1996/07/18 17:38:24  sit
 *     Import from /mit/6.096/share/classes after 6.80s session
 *
 *     Revision 1.3  1996/07/11 14:48:06  sit
 *     Added documentation.
 *
 *     Revision 1.2  1996/07/02 22:33:17  las
 *     Fixed package....
 *
 *     Revision 1.1  1996/07/02 21:47:51  las
 *     Initial revision
 *
 */

