/*
 * cs101 Graphical Boolean semaphore utility
 * $Id: GBS.java,v 1.2 1998/06/03 19:42:00 tparnell Exp $
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see <a href="http://www.ai.mit.edu/projects/cs101">the
 * CS101 homepage</a> or email <las@ai.mit.edu>.
 *
 * Copyright (C) 1996 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */

package cs101.util;

import java.awt.*;

/**
 * cs101.util.GBS implements simple binary semaphores in java.<br>
 * Interface is gbs.request(), gbs.release().<br>
 * It also provides a graphical display of the semaphores status.<br>
 * <p>
 * This utility borrows heavily from the design and implementation
 * of cs101.util.BS.  It mearly adds a graphical display.
 *
 * @author  Todd C. Parnell, tparnell@ai.mit.edu
 * @author  Joshua R. Brown, reuben@ai.mit.edu (added graphics)
 * @author  Lynn Andrea Stein, las@ai.mit.edu (semaphore design)
 * @version $Id: GBS.java,v 1.2 1998/06/03 19:42:00 tparnell Exp $
 * <br>
 * Copyright 1996 Massachusetts Institute of Technology
 *
 */
public class GBS extends GraphicalSemaphore {

  /** The current state of the semaphore.
   *  true => busy , false => free
   */
  private boolean busy; 

  /** The graphical display of the semaphore */
  private ColorField sqr;

  //GBS(boolean, String)
  /**
   * Constructs a binary semaphore with the intial value passed in.
   *
   * @param initval  Initial value for semaphore.  (true => in use)
   * @param label    Used to identfy the semaphore in the display.
   */
  public GBS (boolean initVal, String label) {
    super(label);
    this.busy = initVal;

    // now that everything is initialized setup the GUI
    this.setupGUI();  

  }
  
  /**
   * Does all of the graphical setup on this level
   * then calls the superclasses method to finish the setup.
   *
   * This method is primarly responsible for setting up the 
   * display Panel.
   */
  protected void setupGUI() {
    // Graphics setup on this level
    this.sqr = new ColorField(this.busy, new Dimension(25,25),
			  Color.red, Color.green);

    this.display = new Panel();
    this.display.add(this.sqr);
    
    // Graphics setup in super-class
    super.setupGUI();
  
  }

  /**
   * Requests the semaphore.  If the semaphore is currently busy,
   * causes the requesting process to wait() until the semaphore is
   * release()d.  Unlike java.lang.Object.wait(), the requesting
   * process is not suspended if the semaphore is currently free.
   *
   * @see #release
   * @see java.lang.Object#wait
   */
  synchronized public void request () {
    while (this.busy) {
      try {this.wait();} catch (InterruptedException e) {}
    }
    this.busy = true;
    this.showStatus();
  }

  /**
   * Releases the semaphore.  Any objects currently wait()ing on the
   * semaphore are notify()d (and one of them will be granted the
   * semaphore).  Unlike java.lang.Object.notify(), the semaphore is
   * also freed so that if there are no wait()ing objects, the next
   * object to request() the semaphore will receive it.
   *
   * @see #request
   * @see java.lang.Object#notifyAll()
   */
  synchronized public void release () {
    this.busy = false;
    this.notifyAll();
    this.showStatus();
  }

  /**
   * Prints out the current state of the semaphore.
   * Changes the graphical display.
   */
  protected void showStatus() {      
    // tab over an approprate amount
    for (int i = 0; i<this.myNumber; i++) 
      System.out.print("               ");

    // now print the status 
    System.out.print(this.label.getText()+": ");
    if (this.busy)
      System.out.println("BUSY");
    else 
      System.out.println("FREE");
    
    // update the status field
    this.sqr.changeState(this.busy);
  }

}

/* Comments:
 *
 * History:
 *     $Log: GBS.java,v $
 *     Revision 1.2  1998/06/03 19:42:00  tparnell
 *     update from Java 1.0 to 1.1
 *
 *     Revision 1.1  1998/03/13 22:18:13  tparnell
 *     Import from server crash.  I think the src and class files match up.
 *
 *     Revision 1.5  1996/08/01 18:26:22  reuben
 *     More javadoc tweaking (hopefully the final pass)
 *
 *     Revision 1.4  1996/07/30 17:25:59  reuben
 *     Added/corrected javadoc comments.
 *
 *     Revision 1.3  1996/07/25 18:27:41  reuben
 *     Added all kinds of comments.
 *     Compiled and tested.
 *
 *     Revision 1.2  1996/07/25 15:33:27  reuben
 *     testing
 *
 */
