/*
 * Networked Wire, Client Side
 * 
 * $Id: ClientWire.java,v 1.2 1998/06/24 21:28:53 tparnell Exp $
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see <a href="http://www.ai.mit.edu/projects/cs101">the
 * CS101 homepage</a> or email <las@ai.mit.edu>.
 *
 * Copyright (C) 1996 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */
package cs101.util;

import java.io.*;
import java.net.*;

/**
 * Networked Wire, Client Side.  Provides readObject, writeObject.
 * <p>
 *
 * If server's hostName and port are not provided at creation time, the user
 * is prompted for this information using ClientDialog.
 *
 * @see      cs101.util.Wire
 * @see      cs101.util.ServerWire
 *
 * @author   Todd C. Parnell, tparnell@ai.mit.edu
 * @author   Maciej Stachowiak, maciej@ai.mit.edu 
 * @author   Lynn Andrea Stein, las@ai.mit.edu
 * @version  $Id: ClientWire.java,v 1.2 1998/06/24 21:28:53 tparnell Exp $
 * <br>
 * Copyright 1996 Massachusetts Institute of Technology
 *
 */

public class ClientWire implements Wire {

  /** The network connection.  Contains all of the relevant connect
   * info, if we need it 
   */
  private Socket sock;
  /** Where to read from */ 
  private ObjectInputStream in;
  /** Where to write to */
  private ObjectOutputStream out;
  /** Who we're connected to */
  private String hostname;
  /** Which port we're connected to */
  private int port;

  /**
   *How to make one, if we know who we want to talk to.
   *
   * @param hostName  the name of the machine that the server is on
   * @param port      the port number on which the server is listening
   */
  public ClientWire( String hostName, int port ) {
    this.connectTo( hostName, port );
  }
  /**
   * How to make one, if we don't know who we want to talk to.
   * Uses ClientDialog to ask user.
   */
  public ClientWire() {    
    ClientDialog qd = new ClientDialog();
    qd.ask();
    this.connectTo( qd.getHostName(), qd.getPort() );
  }

  /**
   * Opens a connection to a server presumed to be listening on hostName,
   * port.  Sets up listener thread.  Called by constructor; should not be
   * called otherwise.
   *
   * @param hostName  the name of the machine that the server is on
   * @param port      the port number on which the server is listening
   */
  protected void connectTo( String hostName, int port ) {
    System.out.println("Client:  trying to connect to " + hostName
		       + " on port " + port );
    try {
      this.hostname = hostname;
      this.port = port;
      this.sock = new Socket( hostName, port );
      this.in = new ObjectInputStream( this.sock.getInputStream() );
      this.out = new ObjectOutputStream( this.sock.getOutputStream() );
    } catch (IOException e) {
      throw new RuntimeException("Client:  " +
				 "can't establish communication with " + 
				 hostName + " on port " + port );
    }
  }

  /**
   * Use this to read an Object from the Wire.
   *
   * @returns the Object read.
   */
  public Object readObject() {
    try {
      return this.in.readObject();
    }
    catch (Exception e) {
      throw new RuntimeException("ClientWire:  failed to read from " + 
				 hostname + " on " + port); 
    }
  }

  /**
   * Use this method to write an Object to the Wire.
   *
   * @param o The object to be written.
   */
  public void writeObject( Object o ) {
    try {
      this.out.writeObject( o );
    } catch (IOException e) {
      throw new RuntimeException("ClientWire:  failed to write to " + 
				 hostname + " on " + port); 
    }
  }

  /**
   *  Closes the Socket and Streams.
   */
  public void finalize() {
    try {
      this.in.close();
      this.out.close();
      this.sock.close();
    } catch (IOException e) {}
  }
    
}


/* Comments:
 *
 * History:
 *     $Log: ClientWire.java,v $
 *     Revision 1.2  1998/06/24 21:28:53  tparnell
 *     code cleanup
 *
 *     Revision 1.1  1998/06/24 16:32:16  tparnell
 *     changes after summer98.  added a multi-user Group server and some
 *     other misc files
 *
 *
 */


