//GroupServer.java

package cs101.util;
import java.net.*;
import java.util.*;
import java.io.*;

/** GroupServer implements a central communications server that keeps track of several
 * independent groups.  There should be one GroupServer running on a single machine, and
 * all GroupClients should connect to it.  GroupServer can act as an echoer.
 */
public class GroupServer extends Server
{
	/** The list of groups.  The group ClientGroup.noGroup is <b>not</b> in this list */
	protected Vector groups;
	/** Client command to get server to list the current groups.  Clients should send
	 * this when they first start up, and whenever they receive a GROUPS_CHANGED message */
	public static final String GET_GROUPS= "!!!GET_GROUPS";
	/** Client prefix to join a group.  The server response will be JOINED, at which
	 * point the client will be directly connected with the group. */
	public static final String JOIN_GROUP= "!!!JOIN_GROUP";
	/** Client command to leave a group and go back into the group selection state */
	public static final String LEAVE_GROUP= "!!!LEAVE_GROUP";
	/** Server message to client that it should select a new group */
	public static final String SELECT_GROUP= "!!!\nSELECT_GROUP";
	/** Server message to clients that someone has joined or left a group.  Only sent
	 * to clients who are in the selection state. */
	public static final String GROUPS_CHANGED= "!!!GROUPS_CHANGED";
	/** Client command to notify server that the client wants echoes of messages it sent */
	public static final String WANT_ECHO= "!!!WANT_ECHO";
	/** Client command to notify server that the client does not want echoes of
	 * messages it sent. */
	public static final String DONT_ECHO= "!!!DONT_ECHO";
	/** Server prefix before a "\n"-separated list of group names.  Each group name
	 * consists of the name of the group followed by the number of members in parentheses:
	 * <br><tt>mygroup (2)</tt> */
	public static final String GROUP_LIST= "!!!GROUP_LIST";
	/** Server prefix telling client that it (or someone else) has successfully joined 
	 * a group.  User ID follows. */
	public static final String JOINED= "!!!JOINED";
	/** Server prefix telling client that some other client has left its group.
	 * userID followes */
	public static final String CLIENT_LEFT= "!!!CLIENT_LEFT";
	/** Server message telling client it didn't understand the last command. */
	public static final String FAILED= "FAILED"; 

	/** Create a new GroupServer */
	public GroupServer() {
		super();
		groups= new Vector();
		ClientGroup.noGroup= new ClientGroup("no group", this);
	}
	
	/** Find a group with the given name on the list of groups.
	 * 
	 * @param groupname the name of the group to find
	 * 
	 * @return the group, if one with the appropriate name exists, or <tt>null</tt> if
	 * it doesn't.
	 */
	public ClientGroup findGroup(String groupname) {
		Enumeration e= groups.elements();
		ClientGroup cg;
		while (e.hasMoreElements()) {
			cg= (ClientGroup)e.nextElement();
			if (cg.getName().equals(groupname)) {
				return cg;
			}
		}
		return null;
	}

	/** Join a GroupBabySitter (client) to a group.
	 * @param gbbs the GroupBabySitter (client) to attach to a group
	 * @param groupname the name of the group to add the client to.
	 */
	public synchronized void joinGroup(GroupBabySitter gbbs, String groupname) {
		ClientGroup cg= findGroup(groupname);
		if (cg==null) {
			System.out.println("Creating group: "+groupname);
			cg= new ClientGroup(groupname, this);
			groups.addElement(cg);
//			ClientGroup.noGroup.sendToGroup(this.GROUPS_CHANGED, gbbs);
		}
		gbbs.setGroup(cg);
		cg.sendToGroup(this.JOINED+gbbs.getUserID(), null);
		ClientGroup.noGroup.sendToGroup(this.GROUPS_CHANGED, gbbs);
	}

	/** Removes a group from the list of groups.  Do not call this yourself -- it
	 * is called by ClientGroup.  The group must be empty.
	 * @param g the group to remove.  It <b>must</b> be empty.
	 */
	public synchronized void removeGroup(ClientGroup g) {
		System.out.println("Removing group: "+g.groupName);
		groups.removeElement(g);
//		ClientGroup.noGroup.sendToGroup(this.GROUPS_CHANGED, null);
	}

	/** Create a new GroupBabySitter to watch over a client connection
	 * @param s the Socket to watch
	 */
	protected void spawnBabySitter(Socket s) {
		GroupBabySitter gbbs= new GroupBabySitter(s, (Server)this);
		babySitters.addElement(gbbs);
	}

	/** returns a &lt;cr&gt;-separated list of group names with membership information.
	 * The string ends with a &lt;cr&gt;.  Each line contains the name of the group,
	 * a space, and the number of members in parentheses.
	 */
	protected String getGroupList() {
		StringBuffer sb= new StringBuffer();
		Enumeration e= groups.elements();
		ClientGroup cg;
		while (e.hasMoreElements()) {
			cg= (ClientGroup)e.nextElement();
			sb.append(cg.getName()+" ("+cg.size()+")\n");
		}
		return sb.toString();
	}
}

/** Represents a group of clients who talk to each other. */
class ClientGroup {
	Vector babySitters;
	String groupName;
	GroupServer server;
	public static ClientGroup noGroup;

	/** Create a new ClientGroup with the given name, attached to the server.
	 * @param name the name of the new client group
	 * @param server the server
	 */
	public ClientGroup(String name, GroupServer server) {
		groupName= name;
		babySitters= new Vector();
		this.server= server;
	}
	
	/** return the number of members in this group */
	public int size() {
		return babySitters.size();
	}
	
	/** add a member to this group. */
	public void addClient(GroupBabySitter gbbs) {
		System.out.println("Adding client to group: "+groupName);
		babySitters.addElement(gbbs);
	}
	
	/** remove a member from this group.  If the group becomes empty, this method will
	 * call GroupServer.removeGroup() to destroy the group.
	 */
	public void removeClient(GroupBabySitter gbbs) {
		System.out.println("Removing client from group: "+groupName);
		babySitters.removeElement(gbbs);
		sendToGroup(GroupServer.CLIENT_LEFT+gbbs.getUserID(), gbbs);
		if (this!=noGroup && babySitters.size()==0) {
			server.removeGroup(this);
		}
		if (this!=noGroup) {
			noGroup.sendToGroup(GroupServer.GROUPS_CHANGED, null);
		}
	}
	
	/** return the name of the group */
	public String getName() {
		return groupName;
	}
	
	/** send a message to every member of the group, except for the sender, if the
	 * sender has requested not to receive its own messages (GroupServer.DONT_ECHO).
	 * @param s the message
	 * @gbbs the client that originated the message
	 */
	protected void sendToGroup(String s, GroupBabySitter gbbs) {
		Enumeration e= babySitters.elements();
		GroupBabySitter sendto;
		while (e.hasMoreElements()) {
			sendto= (GroupBabySitter)e.nextElement();
			if (sendto!=gbbs || sendto.wantsReflection()) {
				sendto.send(s);
			}
		}
	}
}

/** This class watches over a client connection */
class GroupBabySitter extends BabySitter
{
	/** the group to which this client belongs */
	protected ClientGroup groupID= null;
	/** true iff this client wants its own messages echoed back to itself. */
	protected boolean noteToSelf= false;
	/** unique (to this server) userID for this babysitter */
	protected String userID= null;
	private static int count=0;

	/** create a new GroupBabySitter to watch over a given socket on a server
	 * @param s the socket to watch
	 * @param the server
	 */
	public GroupBabySitter(Socket s, Server server) {
		super(s, server);
		userID= String.valueOf(count++);
	}
	
	public GroupBabySitter(Socket s, Server server, String userID) {
		super(s, server);
		this.userID= userID;
	}
	
	/** returns the userID of this client
	 */
	public String getUserID() {
		return userID;
	}
	
	/** return true if this client wants to receive its own messages */
	public boolean wantsReflection() {
		return noteToSelf;
	}
	
  /** 
   * Recieves new info from clients.
   *
   * @see Server#sendToAllExcept
   */
	public void run() {
		System.out.println("Server:  BabySitter running");
		setGroup(ClientGroup.noGroup);
		while(true) {
			try {
				String s=is.readUTF();
//				System.out.println( "Server:  just read '" + s +"' from " +
//					this.sock.getInetAddress().getHostName() +
//					", port " + this.sock.getPort() );

				if( s==null || s=="" ) {
					this.server.removeBabySitter(this);
					return;
				}
				if (this.groupID==ClientGroup.noGroup) {
					processGroupQuery(s);
				} else if (s.equals(GroupServer.LEAVE_GROUP)) {
					setGroup(ClientGroup.noGroup);
					send(GroupServer.SELECT_GROUP);
				} else {
					groupID.sendToGroup(s, this);
				}
			} catch (IOException e) {
				System.out.println("Server:  socket error in run ");
				this.server.removeBabySitter(this);
			} 		
		}
	}
	
	/** Send a message to this one client
	 * @param s the message
	 */
	protected synchronized void send(String s) {
		try {
			//	    System.out.println("Server:  sending "+s);
			os.writeUTF(s);
			//	    System.out.println("Server:  successfully sent "+s);
		} catch (IOException e) {
			System.out.println("Server:  socket error in send");
			this.server.removeBabySitter(this);
		}
	}

	/** halts the GroupBabySitter, and removes it from its group */
	protected void stop() {
		groupID.removeClient(this);
		super.stop();
	}

	/** remove this client from its old group, and place it in a new one
	 * @param g the new group
	 */
	protected void setGroup(ClientGroup g) {
		if (groupID!=null) {
			groupID.removeClient(this);
		}
		groupID= g;
		groupID.addClient(this);
	}
	
	/** handle the input if this client is in ClientGroup.noGroup.  This happens when
	 * the client is in the selection state.
	 */
	protected void processGroupQuery(String s) {
		if (s.equals(GroupServer.GET_GROUPS)) {
			send(GroupServer.GROUP_LIST+((GroupServer)server).getGroupList());
		} else if (s.startsWith(GroupServer.JOIN_GROUP)) {
			groupID.removeClient(this);
			groupID= null;
			String join= s.substring(GroupServer.JOIN_GROUP.length());
			((GroupServer)server).joinGroup(this, join);
//			send(GroupServer.JOINED+userID);
		} else if (s.equals(GroupServer.WANT_ECHO)) {
			this.noteToSelf= true;
		} else if (s.equals(GroupServer.DONT_ECHO)) {
			this.noteToSelf= false;
		} else {
			send(GroupServer.FAILED);
		}
	}
		
}