/*
 * SimulationPanel.java
 * Part of the BinSort problem set.
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see http://www.ai.mit.edu/projects/cs101, the
 * CS101 homepage or email las@ai.mit.edu.
 *
 * Copyright (C) 1997 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */

package BinSort;

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;

class SimulationPanel extends Panel implements Runnable {

    ControlPanel my_control;
    Image offscreen_image;
    Graphics offscreen_graphics;
    int my_height = 0, my_width = 0;
    
    private ThreadGroup my_threadgroup;
    private Vector nodes, channels;
    /*private*/ boolean running;
    /*private*/ BinSortElement selected_bse = null;
    
    private Thread spirit;
    
    /*private*/ int repaint_pause = 200, numNodes = 0;

    /*private*/ BinSortElement anchor;
    Point anchorPoint, currentPoint;
    
    PopupMenu popup;
    
    protected static final Color COLOR[] = new Color[5];
    static {
	COLOR[0] = Color.blue;
	COLOR[1] = Color.green;
	COLOR[2] = Color.yellow;
	COLOR[3] = Color.pink;
	COLOR[4] = Color.cyan;
    }

    
    public SimulationPanel(ControlPanel my_c) {
	super();
	this.my_control = my_c;
	this.nodes = new Vector();
	this.channels = new Vector();
	
	this.spirit = new Thread(this);
	spirit.start();
	
	my_threadgroup = new ThreadGroup("control threadgroup");
	stopSimulation();
	
	switchSelected(null);
	
	popup = new PopupMenu();
	
	
	this.addMouseMotionListener(new MouseMotionAdapter() {
	    public void mouseMoved(MouseEvent e) {
		if (!my_control.channelSelected() || 
			anchorPoint == null) return;
		currentPoint = e.getPoint();
	    }
	});

	this.addMouseListener(new MouseAdapter() {
	    public void mouseClicked(MouseEvent e) {
		//pre-lim setup (common)
		Point temp = e.getPoint();
		BinSortElement bse = null;
		try { 
		    bse = whatsInside(temp); 
		    switchSelected(bse);
		}
		catch (IsInsideNothingException exc1) {} // bse == null
		
		if (!my_control.channelSelected) { // node selected in CP
		    anchorPoint = null;
		    anchor = null;
		    // done if we selected a node and clicked on anything
		    if (bse != null) return; 
		    // otherwise, create a new node
		    try {
			Node newNode = my_control.getNewNode(SimulationPanel.this, my_threadgroup);
			switchSelected(newNode);
			((Node)selected_bse).setPos(temp);
			selected_bse.setSelected(true);
			if (SimulationPanel.this.running) { ((Node)selected_bse).start(); }
			nodes.addElement(selected_bse);
			notifyMenus();
			return;
		    }
		    catch (NoNodeSelectedException exc2) { return; }
		}
		// channel selected in CP
		
		if (e.getClickCount() > 1) { // double click to anchor
		    try {
			anchor = (Node)bse;
			anchorPoint = ((Node)anchor).getPos(); 
		    }
		    catch (ClassCastException cce1) {} // anchor is channel
		    catch (NullPointerException npe1) {} // nothing selected
		    finally { return; }
		}
			
		if (bse == null || anchor == null) {
		    anchorPoint = null;
		    anchor = null;
		    return;
		}
			
		// create a new channel
		Node n1 = (Node)SimulationPanel.this.anchor;
		Node n2 = (Node)bse;
		if (channelAlreadyExists(n1, n2)) return;
		try {
		    Channel bsc = new Channel(n1, n2, SimulationPanel.this, my_threadgroup);
		    channels.addElement(bsc);
		    switchSelected(bsc);
		    if (SimulationPanel.this.running) { ((Channel)selected_bse).start(); }
		    anchorPoint = null;
		    anchor = null;
		}
		catch(SameNodesException exc3) { 
		    anchorPoint = null; 
		    anchor = null;
		}
	    }
	});
    }
    

    // parsing something into a SimulationPanel
    public static SimulationPanel parse(String sim_str, int[] ref_array, Class[] node_classes, ControlPanel cp) throws ParseException{

	SimulationPanel sp = new SimulationPanel(cp);
	StringTokenizer st = new StringTokenizer(sim_str, "|||");
	if (st.countTokens() != 2) throw(new ParseException("incorrect string format"));
	String nodes_str = st.nextToken();
	String channels_str = st.nextToken();

	// add the nodes
	StringTokenizer n_st = new StringTokenizer(nodes_str,"\n");
	String item;
	StringTokenizer item_st;
	while (n_st.hasMoreTokens()) {
	    item = n_st.nextToken();
	    item_st = new StringTokenizer(item, " ");
	    if (item_st.countTokens() !=3) throw(new ParseException("incorrect string format"));
	    String type;
	    int x_coord = 0, y_coord = 0;
	    try {
		type = item_st.nextToken();
		x_coord = Integer.parseInt(item_st.nextToken());
		y_coord = Integer.parseInt(item_st.nextToken());
	    }
	    catch(NumberFormatException e) {
		throw(new ParseException("incorrect string format"));
	    }
	    try {
		int type_int = Integer.parseInt(type);
		NodeBehavior new_node_b = (NodeBehavior)node_classes[ref_array[type_int]].newInstance();
		Node new_node = new Node(new_node_b, sp, sp.getThreadGroup(), COLOR[ref_array[type_int]]);
		sp.nodes.addElement(new_node);
		new_node.setPos(x_coord, y_coord);
	    }
	    catch(NumberFormatException exc) {
		if ((!type.equals("g")) && (!type.equals("t"))) throw(new ParseException());
		Node new_node;
		if (type.equals("g")) {
		    new_node = new GeneratorNode(sp, sp.getThreadGroup());
		} else {
		    new_node = new TerminatorNode(sp, sp.getThreadGroup());
		}
		sp.nodes.addElement(new_node);
		new_node.setPos(x_coord,y_coord);
	    }
	    catch(ArrayIndexOutOfBoundsException exc) {
		throw(new ParseException());
	    }
	    catch(ClassCastException exc) {
		throw(new ParseException());
	    }
	    catch(IllegalAccessException exc) {
		throw(new ParseException());
	    }
	    catch(InstantiationException exc) {
		throw(new ParseException());
	    }
	}

	// now add the channels
	StringTokenizer c_st = new StringTokenizer(channels_str,"\n");
	while(c_st.hasMoreTokens()) {
	    item = c_st.nextToken();
	    item_st = new StringTokenizer(item, " ");
	    if (item_st.countTokens() != 2) throw(new ParseException());
	    int start_n = 0, end_n = 0;
	    try {
		start_n = Integer.parseInt(item_st.nextToken());
		end_n = Integer.parseInt(item_st.nextToken());
	    }
	    catch(NumberFormatException e) {
		throw(new ParseException());
	    }
	    try {
		Channel new_channel = new Channel((Node)sp.nodes.elementAt(start_n), (Node)sp.nodes.elementAt(end_n), sp, sp.getThreadGroup());
		sp.channels.addElement(new_channel);
	    }
	    catch(SameNodesException e) {
		throw(new ParseException());
	    }
	}

	return(sp);
    }

    public String unparse(Class[] n_types) {
	String total_str = "";

	// print out the nodes
	Enumeration enum = this.nodes.elements();
	String node_type_string = "";
	while(enum.hasMoreElements()) {
	    Node n = (Node)enum.nextElement();
	    if (n instanceof GeneratorNode) node_type_string = "g";
	    else
		if (n instanceof TerminatorNode) node_type_string = "t";
	    else
		for (int i = 0; i < n_types.length; i++) {
		    if (n.getBehavior().getClass() == n_types[i]) {
			node_type_string = Integer.toString(i);
			break;
		    }
		}
	    total_str += node_type_string + " " + n.getPos().x + " " + n.getPos().y + "\n";
	}

	// separator
	total_str += "|||\n";
    
	// and now, print out the channels
	enum = this.channels.elements();
	while(enum.hasMoreElements()) {
	    Channel bsc = (Channel)enum.nextElement();
	    int start_pos = this.nodes.indexOf(bsc.getStartNode());
	    int end_pos = this.nodes.indexOf(bsc.getEndNode());
	    total_str += Integer.toString(start_pos) + " " + Integer.toString(end_pos) + "\n";
	}

	return(total_str);
    }
	
    public boolean isRunning() { return running; }
    
    public Dimension getMinimumSize() { return(this.getPreferredSize()); }

    public Dimension getPreferredSize() {
    	Enumeration enum = this.nodes.elements();
	int x = 700, y = 350;
	while(enum.hasMoreElements()) {
	    Node n = (Node)enum.nextElement();
	    if (n.getPos().x + 50 > x) x = n.getPos().x + 50;
	    if (n.getPos().y + 75 > y) y = n.getPos().y + 75;
	}
	return(new Dimension(x, y));
    }
    
    /*private*/ void switchSelected(BinSortElement bse) {
	if (selected_bse != null) selected_bse.setSelected(false);
	selected_bse = bse;
	if (selected_bse != null) selected_bse.setSelected(true);
	notifyMenus();
    }
    
    //this needs to be cleaned up.
    /*private*/ void notifyMenus() {
	if (selected_bse != null) {
//	    if ( (getSelectedElement()).isEditable() )
//		my_control.myMain.configure_item.setEnabled(true);
//	    else my_control.myMain.configure_item.setEnabled(false);
	    my_control.myMain.enable_item.setEnabled(!selected_bse.isEnabled());
	    my_control.myMain.disable_item.setEnabled(selected_bse.isEnabled());
	    my_control.myMain.delete_item.setEnabled(true);
	}
	else {
//	    my_control.myMain.configure_item.setEnabled(false);
	    my_control.myMain.enable_item.setEnabled(false);
	    my_control.myMain.disable_item.setEnabled(false);
	    my_control.myMain.delete_item.setEnabled(false);
	}    
    }
    
    /*private*/	boolean isInsideAnything(Point p) {
	try {
	    BinSortElement bse = whatsInside(p);
	    return true;
	}
	catch(IsInsideNothingException e) { return false; }
    }
    
    /*private*/ BinSortElement whatsInside(Point p) throws IsInsideNothingException {

	BinSortElement bse;
	Enumeration enum;
	enum = nodes.elements();
	while (enum.hasMoreElements()) {
	    bse = (BinSortElement)enum.nextElement();
	    if (bse.isInside(p)) return(bse);
	}

	enum = channels.elements();
	while (enum.hasMoreElements()) {
	    bse = (BinSortElement)enum.nextElement();
	    if (bse.isInside(p)) return(bse);
	}
	throw(new IsInsideNothingException());
    }

    /*private*/ boolean channelAlreadyExists(Node a, Node b) {
	Enumeration enum = channels.elements();
	Channel ch;
	while (enum.hasMoreElements()) {
	    ch = (Channel)enum.nextElement();
	    if (((ch.getStartNode()) == a) && ((ch.getEndNode()) == b)) {
		    return(true);
	    }
	}
	return(false);
    }

    /*private*/ void paintOffscreen() {

	Dimension d = this.getSize();
	if ((my_height != d.height) || (my_width != d.width)) {
	    my_height = d.height;
	    my_width = d.width;
	    offscreen_image = createImage(my_width, my_height);
	}

	Enumeration enum;

	// get the graphics
	offscreen_graphics = offscreen_image.getGraphics();

	// Paint the background
	offscreen_graphics.setColor(Color.white);
	offscreen_graphics.fillRect(0, 0, my_width, my_height);
	
	// Paint the channels
	enum = channels.elements();
	while (enum.hasMoreElements()) {
	    ((Channel)(enum.nextElement())).paint(offscreen_graphics);
	}

	// Paint the nodes
	enum = nodes.elements();
	while (enum.hasMoreElements()) {
	    ((Node)(enum.nextElement())).paint(offscreen_graphics);
	}

	// Paint the extra stuff
	if (anchorPoint != null && currentPoint != null) {
	    offscreen_graphics.setColor(Color.red);
	    offscreen_graphics.drawLine(anchorPoint.x, anchorPoint.y, 
					    currentPoint.x, currentPoint.y);
	}

    }
    
    void loadNetwork() {}
    void saveNetwork() {}

    void clearAll() {
	this.nodes.removeAllElements();
	this.channels.removeAllElements();
	this.my_threadgroup.suspend();
    }
    
    void startSimulation() {
	Enumeration enum;
	enum = nodes.elements();
	while (enum.hasMoreElements()) { ((Node)enum.nextElement()).start(); }
	enum = channels.elements();
	while (enum.hasMoreElements()) { ((Channel)enum.nextElement()).start(); }
	repaint_pause = 100;
	my_control.setRunning(true);
	running = true;
    }

    void stopSimulation() {
	Enumeration enum;
	enum = nodes.elements();
	while (enum.hasMoreElements()) { ((Node)enum.nextElement()).stop(); }
	my_control.setRunning(false);
    	enum = channels.elements();
	while (enum.hasMoreElements()) { ((Channel)enum.nextElement()).stop(); }
	repaint_pause = 500;
	running = false;
    }

    void deleteSelection() {
	// let's make sure nothing evil happens, we suspend all threads
	my_threadgroup.suspend();
	try { selected_bse.destroy(); }
	catch(NullPointerException e) {}
	my_threadgroup.resume();
    }

    void enableSelection() {
	if (selected_bse == null) return; 
	this.selected_bse.setEnabled(true);
    }

    void disableSelection() {
	if (selected_bse != null) this.selected_bse.setEnabled(false);
    }
    
    void configureSelection() { selected_bse.configure(); }
    
    ThreadGroup getThreadGroup() { return my_threadgroup; }
    
    void notifyOfDestruction(BinSortElement bse) {
	try { notifyOfDestruction((Node)bse); }
	catch(ClassCastException e) {}
	try { notifyOfDestruction((Channel)bse); }
	catch(ClassCastException e) {}
    }
    
    void notifyOfDestruction(Node n) {
	boolean b = nodes.removeElement(n);
    }

    void notifyOfDestruction(Channel c) {
	boolean b = channels.removeElement(c);
    }
	
    public BinSortElement getSelectedElement() { return selected_bse; }
    
    public void update(Graphics g) { paint(g); }
    
    public void paint(Graphics g) {
	paintOffscreen();
	g.drawImage(offscreen_image, 0, 0, this);
    }

    public void run() {
	try {
	    while(true) {
		Thread.sleep(this.repaint_pause);
		repaint();
	    }
	}
	catch(InterruptedException e){}
    }
    
}
