/*
 * Channel.java
 * Part of the BinSort problem set.
 *
 * Developed for "Rethinking CS101", a project of Lynn Andrea Stein's AP Group.
 * For more information, see http://www.ai.mit.edu/projects/cs101, the
 * CS101 homepage or email las@ai.mit.edu.
 *
 * Copyright (C) 1997 Massachusetts Institute of Technology.
 * Please do not redistribute without obtaining permission.
 */

package BinSort;

import java.awt.*;
import java.awt.event.*;

public class Channel implements BinSortElement, InputChannel, OutputChannel {
    
    static private ChannelFrame cf;
    
    private Thread spirit;
    private ThreadGroup tg;
    
    private Queue channel_objects, timeStamps;
    private int timeCounter = 0;
    
    private float mvt_constant = 0;

    private int capacity = 9, latency = 1;

    private String name;
    private SimulationPanel my_sp;
    
    private Node startNode, stopNode;

    private boolean selected, enabled = true, full, forDisplayOnly, destroyed;
    
    // for painting performance
    private Point p1, p2;
    private Polygon poly1, poly2;
    private int x_diff, y_diff;
    
    public Channel(Node start, Node end, SimulationPanel sp, ThreadGroup ntg) throws SameNodesException {
	if (start == end) throw(new SameNodesException());
	this.name = "**UNNAMED**";
	my_sp = sp; startNode = start; stopNode = end;
	this.tg = ntg;
	channel_objects = new Queue();
	timeStamps = new Queue();
	startNode.addOutputChannel((OutputChannel)this);
	stopNode.addInputChannel((InputChannel)this);
	
	setupGUI();
    }
    
    private void setupGUI() {
	p1 = startNode.getPos();
	p2  = stopNode.getPos();
	if (p1 == null || p2 == null) return;
	int mid_x = (p1.x + p2.x) / 2;
	int mid_y = (p1.y + p2.y) / 2;
	x_diff = p2.x - p1.x;
	y_diff = p2.y - p1.y;
	float dist = (int)Math.sqrt(x_diff * x_diff + y_diff * y_diff);
	int x_ratio = (int)Math.round((y_diff * 3) / dist);
	int y_ratio = (int)-Math.round((x_diff * 3) / dist);

	poly1 = new Polygon();
	poly1.addPoint(p1.x - x_ratio, p1.y - y_ratio);
	poly1.addPoint(p1.x + x_ratio, p1.y + y_ratio);
	poly1.addPoint(p2.x + x_ratio, p2.y + y_ratio);
	poly1.addPoint(p2.x - x_ratio, p2.y - y_ratio);
	
	poly2 = new Polygon();
	poly2.addPoint(mid_x - 3 * x_ratio, mid_y - 3 * y_ratio);
	poly2.addPoint(mid_x + 3 * x_ratio, mid_y + 3 * y_ratio);
	poly2.addPoint(p1.x + x_diff * 7 / 10, p1.y + y_diff * 7 / 10);
    }

    
// Channel Stuff
    public void writeObject(Object o) throws ChannelFullException, ChannelDisabledException {
	if (channel_objects.size() >= capacity) {
	    full = true;
	    throw(new ChannelFullException());
	}
	if (!enabled) throw(new ChannelDisabledException());
	channel_objects.addObject(o);
	timeStamps.addObject(new Integer(timeCounter));
	mvt_constant = 0;
    }
    
    public Object readObject() throws ChannelEmptyException, ChannelDisabledException {
	if (!enabled) throw(new ChannelDisabledException());
	if (channel_objects.size() < capacity - 1) full = false;
	
	int time;
	try {
	    time = ((Integer)timeStamps.peekObject()).intValue();   
	    if (time > timeCounter) time -= 1000;
	    if (timeCounter - time <= latency) throw(new ChannelEmptyException());
	    mvt_constant = 0.8f;
	    time = ((Integer)timeStamps.getObject()).intValue();
	    return(channel_objects.getObject());
	}
	catch(EmptyQueueException e) { 
	    throw(new ChannelEmptyException());
	}
    }
    
    Node getStartNode() { return startNode; }
    Node getEndNode() { return stopNode; }
    
    public int getCapacity() { return capacity; }
    public void setCapacity(int cap) {
	if (cap > 50) cap = 50;
	if (cap < 0)  cap = 0;
	capacity = cap;
    }
    
    public int getLatency() { return latency; }
    public void setLatency(int lat) {
	if (lat > 50) lat = 50;
	if (lat < 0) lat = 0;
	latency = lat;
    }

//BSE Stuff
    
    public void setSelected(boolean b) { selected = b; }
    public boolean isSelected() { return(selected); }
    
    public void setEnabled(boolean b) { 
	if (forDisplayOnly) return;
	enabled = b;
    }
    public boolean isEnabled() { return(enabled); }

    public boolean isInside(int x, int y) { return(poly1.contains(x, y)); }
    public boolean isInside(Point p) { return(this.isInside(p.x, p.y)); }
    
    public void setDisplayOnly() {  
	if (spirit != null) spirit.stop();
	spirit = null;
	forDisplayOnly = true;
	enabled = false;
    }
    
    public void setName(String n) { this.name = n; }
    public String getName() { return this.name; }

    public void destroy() {
	if (destroyed) return;
	setDisplayOnly();
	destroyed = true;
	startNode.notifyOfDestruction(this);
	stopNode.notifyOfDestruction(this);
	my_sp.notifyOfDestruction(this);
    }
    
    public void notifyOfDestruction(BinSortElement bse) {
	destroy();
    }
    
    public boolean isEditable() { return true; }
    public void configure() { 
	if (cf == null) cf = new ChannelFrame();
	cf.setChannel(this); 
	cf.show();
    }
    
    /* I've added some performance hacks, as a profiler I ran says
     * the old version of paint was a complete dog.  Filling polygons
     * is still very slow. -- Todd
     */
    public void paint(Graphics g) {
	if (p1 == null || p2 == null) setupGUI();
	if (my_sp.isRunning()) mvt_constant += .1f;
	if (mvt_constant >= 1) mvt_constant = 0;
    
	g.setColor(Color.black);
	if (full) g.setColor(Color.red);
	if (!enabled) g.setColor(Color.gray);
	g.fillPolygon(poly1);
	g.fillPolygon(poly2);
	if (this.selected) {
	    g.setColor(Color.red);
	    g.drawPolygon(poly2);
	}

	g.setColor(Color.red);
	if (full) g.setColor(Color.pink);
	if (!enabled) g.setColor(Color.black);
	int j = channel_objects.size();
	for (int i = 1; (i <= j) && ( i<= 6); i++) {
	    int start_x = p1.x + (int)(x_diff * (i + 2 + mvt_constant) / 12);
	    int start_y = p1.y + (int)(y_diff * (i + 2 + mvt_constant) / 12);
	    g.fillOval(start_x, start_y, 8, 8);
	}
    }
    
    public void start() { 
	if (forDisplayOnly || !enabled) return;
	if (spirit == null) {
	    spirit = new Thread(tg, this);
	    spirit.start();
	}
	this.spirit.resume();
    }
    
    public void stop()  { 
	if (forDisplayOnly || !enabled || spirit == null) return;
	this.spirit.suspend();
    }

    public void run() {
	while (true) {
	    try { Thread.sleep(200); }
	    catch(InterruptedException e) {}
	    timeCounter = ++timeCounter % 10000;
	}
    }
    
    public class ChannelFrame extends Frame {
	
	Channel myChannel;
	TextField bandwidth, latency;
	Button ok, apply, cancel;
	
	public ChannelFrame() {
	    
	    setTitle("Channel Configuration");
	    Panel p = new Panel();
	    p.setLayout(new FlowLayout());
	    p.add(new Label("Bandwidth [0-50]:"));
	    bandwidth = new TextField(4);
	    p.add(bandwidth);
	    p.add(new Label(" Latency [0-50]:"));
	    latency = new TextField(4);
	    p.add(latency);
	    add("Center", p);
	    
	    Panel q = new Panel();
	    q.setLayout(new FlowLayout());
	    ok = new Button("OK");
	    ok.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    configureChannel();
		    ChannelFrame.this.setVisible(false);
		}
	    });
	    apply = new Button("Apply");
	    apply.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    configureChannel();
		}
	    });
	    cancel = new Button("Cancel");
	    cancel.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    ChannelFrame.this.setVisible(false);
		}
	    });
	    q.add(ok);
	    q.add(apply);
	    q.add(cancel);
	    add("South", q);
	    
	    this.pack();
	    this.show();
	    
	    this.addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    ChannelFrame.this.setVisible(false);
		}
	    });
	}
	
	void setChannel(Channel newChannel) { 
	    myChannel = newChannel;
	    setTextFields();
	}
	
	void setTextFields() {
	    bandwidth.setText(Integer.toString(getBandwidth()));
	    latency.setText(Integer.toString(getLatency()));
	}
	
	void configureChannel() {
	    if (myChannel == null) return;
	    try {
		myChannel.setCapacity(Integer.parseInt(bandwidth.getText().trim()));
		myChannel.setLatency(Integer.parseInt(latency.getText().trim()));
	    }
	    catch(NumberFormatException e) {}
	}
	
	int getBandwidth() {
	    if (myChannel == null) return 0;
	    return myChannel.getCapacity();
	}
	
	int getLatency() {
	    if (myChannel == null) return 0;
	    return myChannel.getLatency();
	}
    }
}
