/*
 * IntPipe.java
 *
 * Created on September 29, 2005, 11:13 AM
 *
 * (C) 2005 Olin College
 */

package calculator;

import java.util.LinkedList;

/**
 * A blocking queue of ints.  If there are no elements in the pipe when the
 * <code>remove()</code> method is called, <code>remove()</code> will wait
 * until some other thread <code>add()</code>s a new item.
 */
public class IntPipe {
    /** the queue.  It's a list of Integers.
      * (an Integer is an Object subclass containing a single unmodifiable int)
      */
    LinkedList<Integer> queue;
    
    /**
     * Creates a new instance of IntPipe 
     */
    public IntPipe() {
        queue = new LinkedList<Integer>();
    }
    
    /**
     * Adds an item to the queue.  If any consumers were waiting for an item,
     * one will wake up and consume the item.
     */
    public void add(int item) {
        synchronized(queue) {
            // add the item to the back of the list
            queue.addLast(new Integer(item));
            if (queue.size() == 1) {
                // we just added the only item.  Notify a consumer that
                // might be waiting.
                queue.notify();
            }
        }
    }

    /**
     * Removes an item from the queue.  If there are no items to be removed,
     * this method blocks until one becomes available.
     */
    public int remove() {
        synchronized(queue) {
            while (queue.isEmpty()) {
                // continue to wait until something is available
                try {
                    queue.wait();
                } catch (InterruptedException ie) {
                    // do nothing; continue to wait.
                }
            }
            // take the item off the front of the list
            Integer value = queue.removeFirst();
            return value.intValue();
        }
    }
    
}
